/*
 *  LICENSE:
 *  Copyright (c) 2007 Freescale Semiconductor
 *  
 *  Permission is hereby granted, free of charge, to any person 
 *  obtaining a copy of this software and associated documentation 
 *  files (the "Software"), to deal in the Software without 
 *  restriction, including without limitation the rights to use, 
 *  copy, modify, merge, publish, distribute, sublicense, and/or 
 *  sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following 
 *  conditions:
 *  
 *  The above copyright notice and this permission notice 
 *  shall be included in all copies or substantial portions 
 *  of the Software.
 *  
 *  THIS SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
 *  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
 *  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
 *  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 *  
 *
 *  File: $Id: axe-ipc.h,v 1.3 2007/11/27 20:57:20 r21893 Exp $
 *
 */

/**
 *	\file axe-ipc.h
 *	\brief Types and definitions for the PPC-AXE inter-processor communication protocol
 */
 
/**
 *  \addtogroup PPC_AXE_DRIVER
 *  @{
 *
 */
 
#ifndef __AXE_IPC_H
#define __AXE_IPC_H

/*
 * Message types
 */
#define AXE_MT_FIRSTVALID		0

#define AXE_MT_PROCESS_CREATE	0
#define AXE_MT_PROCESS_START	1
#define AXE_MT_PROCESS_STOP		2
#define AXE_MT_PROCESS_KILL		3
#define AXE_MT_PROCESS_STATUS	4
#define AXE_MT_MALLOC			5
#define AXE_MT_FREE				6
#define AXE_MT_CPRINT			7
#define AXE_MT_CSCAN			8
#define AXE_MT_PROCESS_DONE		9
#define AXE_MT_SERVER_LOAD		10
#define AXE_MT_SERVER_UNLOAD	11

#define AXE_MT_LASTVALID		11

#define AXE_MT_UNDELIVERABLE	100

/*
 * Message header word offsets
 */
#define AXE_MSG_PID			0
#define AXE_MSG_SID			1
#define AXE_MSG_TYPE		2
#define AXE_MSG_SIZE		3
#define AXE_MSG_PAYLOAD		4

/*
 * Message payload word offsets and sizes
 */

#define AXE_MSG_PROCESS_CREATE_TCB				0 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_CREATE_RETURN			1 + AXE_MSG_PAYLOAD

#define AXE_MSG_PROCESS_START_PID				0 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_START_RETURN			1 + AXE_MSG_PAYLOAD

#define AXE_MSG_PROCESS_STOP_PID				0 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_STOP_RETURN				1 + AXE_MSG_PAYLOAD

#define AXE_MSG_PROCESS_KILL_PID				0 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_KILL_RETURN				1 + AXE_MSG_PAYLOAD

#define AXE_MSG_PROCESS_STATUS_PID				0 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_STATUS_CURR_STATE		1 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_STATUS_RUN_TIME			2 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_STATUS_READY_TIME		3 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_STATUS_SUSPEND_TIME		4 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_STATUS_MAX_LATENCY		5 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_STATUS_RETURN			6 + AXE_MSG_PAYLOAD

#define AXE_MSG_MALLOC_BYTES					0 + AXE_MSG_PAYLOAD
#define AXE_MSG_MALLOC_REGION					1 + AXE_MSG_PAYLOAD
#define AXE_MSG_MALLOC_RETURN					2 + AXE_MSG_PAYLOAD

#define AXE_MSG_FREE_PTR						0 + AXE_MSG_PAYLOAD
#define AXE_MSG_FREE_RETURN						1 + AXE_MSG_PAYLOAD

#define AXE_MSG_CPRINT_STRING					0 + AXE_MSG_PAYLOAD
#define AXE_MSG_CPRINT_RETURN					1 + AXE_MSG_PAYLOAD

#define AXE_MSG_CSCAN_BUFFER					0 + AXE_MSG_PAYLOAD
#define AXE_MSG_CSCAN_BYTES						1 + AXE_MSG_PAYLOAD
#define AXE_MSG_CSCAN_RETURN					2 + AXE_MSG_PAYLOAD

#define AXE_MSG_PROCESS_DONE_PID				0 + AXE_MSG_PAYLOAD
#define AXE_MSG_PROCESS_DONE_RETURN				1 + AXE_MSG_PAYLOAD

#define AXE_MSG_SERVER_LOAD_IMAGE				0 + AXE_MSG_PAYLOAD
#define AXE_MSG_SERVER_LOAD_PRIORITY			1 + AXE_MSG_PAYLOAD
#define AXE_MSG_SERVER_LOAD_RETURN				2 + AXE_MSG_PAYLOAD

#define AXE_MSG_SERVER_UNLOAD_PID				0 + AXE_MSG_PAYLOAD
#define AXE_MSG_SERVER_UNLOAD_RETURN			1 + AXE_MSG_PAYLOAD

/*
 * hw FIFO definitions
 */
#define AXE_PPC_TX_FIFO 		FIFO1
#define AXE_PPC_RX_FIFO 		FIFO2
#define AXE_PPC_FIFO_EMPTY		0x10
#define AXE_PPC_FIFO_NOTEMPTY	0x08
#define AXE_PPC_FIFO_NOTFULL	0x04
#define AXE_FIFO_SIZE			4


/*
 * Memory Allocation Regions for AxeMalloc API
 */
#define AXE_MALLOC_ANY			0
#define AXE_MALLOC_INTERNAL		1
#define AXE_MALLOC_EXTERNAL		2
#define AXE_MALLOC_IRAM			3

/*
 * Queue message type
 */
typedef struct axeQueueMsg_t
{
	void					*msg;
	struct axeQueueMsg_t	*next;
} axeQueueMsg;


/**
 *	\brief		Allocate a message buffer and initialize type.
 *	\param[in]  
 *	\param[out]	
 *	\return
 *
 *	
 *
 *  \note
 */
unsigned *AxeAllocateMessage(int msg_type);

/**
 *	\brief		Sends the given message without waiting for a reply
 *	\param[in]  
 *	\param[out]	
 *	\return
 *
 *	
 *
 *  \note
 */
int AxeSendMessageWait(void *msg, unsigned waitTime);

/**
 *	\brief		Retrieves the next inbound message from the queue
 *	\param[in]  
 *	\param[out]	
 *	\return
 *
 *	This function returns the next message which originated on the AXE (either
 *	from the Scheduler or a running task).  This will not return messages which 
 *	are replies to messages originally sent by the driver or by a PPC application
 *	through the driver.
 *
 *  \note
 */
int AxeGetInboundMessage(void **msg, unsigned waitTime);

/**
 *	\brief		Retrieves a message reply
 *	\param[in]  msg - pointer-pointer to the original message.
 *	\return
 *
 *	This function receives the reply to the given message.  Since the protocol
 *	specified that a message reply uses the same buffer as the original message,
 *	the reply is identified by the message pointer when it enters the receive FIFO.
 *  No new pointer need be returned to the called of this function.  A successful
 *	return code, AXE_E_OK, indicates that the message was received and removed from
 *	the FIFO and that the original message buffer may now be examined for any results
 *	placed there by the original message recipient.
 *
 *  \note
 */
int AxeGetReplyMessage(void *msg, unsigned waitTime);

/**
 *	\brief		Place a message in the given software message queue
 *	\param[in]  queue - pointer-pointer to the message queue.
 *	\param[in]  msg - pointer to the message.
 *	\return		Success/Fail code
 *
 *  \note
 */
int AxeEnqueueMessage(axeQueueMsg **queue, void *msg);

/**
 *	\brief		Remove a message from the given software message queue
 *	\param[in]  queue - pointer-pointer to the message queue.
 *	\param[in]	matchingMsg - If non-NULL, causes the function to return success
 *				only if a message is found with this pointer value.
 *	\param[in]  matchingPid - filters the result by process Id.  Only messages
 *				addressed to this Id will be returned.  May use AXE_PID_NONE to
 *				allow any message to be returned regardless of recipient Id.
 *	\return		Message pointer from the queue or NULL if not found
 *
 *  \note
 */
void *AxeDequeueMessage(axeQueueMsg **queue, void *matchingMsg, int matchingPid);


/** @} PPC_AXE_DRIVER */
#endif /* __AXE_IPC_H */



/*345678901234567890123456789012345678901234567890123456789012345678901234567890
/*       1         2         3         4         5         6         7         8

/**
 *	\brief	
 *	\param[in]  
 *	\param[out]	
 *	\return
 *
 *	
 *
 *  \note
 */
//void func()
//{
//    
//}
